// SPDX-License-Identifier: GPL-3.0-or-later

//go:build cgo
// +build cgo

package db2

const (
	// Edition detection queries - each targets specific edition characteristics
	queryDetectVersionLUW = `SELECT SERVICE_LEVEL FROM SYSIBMADM.ENV_INST_INFO`
	queryDetectVersionI   = `SELECT 'DB2 for i' FROM SYSIBM.SYSDUMMY1 WHERE EXISTS (SELECT 1 FROM QSYS2.SYSTABLES WHERE TABLE_SCHEMA = 'QSYS2' FETCH FIRST 1 ROW ONLY)`
	queryDetectVersionZOS = `SELECT VERSION FROM SYSIBM.SYSVERSIONS WHERE VERSION_TYPE = 'DB2'`

	queryLockMetrics = `
		SELECT 
			LOCK_WAITS,
			LOCK_TIMEOUTS,
			DEADLOCKS,
			LOCK_ESCALS,
			LOCKS_HELD as LOCK_ACTIVE,
			LOCK_WAIT_TIME,
			LOCKS_WAITING as LOCK_WAITING_AGENTS,
			LOCK_LIST_IN_USE as LOCK_MEMORY_PAGES,
			TOTAL_SORTS,
			SORT_OVERFLOWS,
			ROWS_READ,
			ROWS_MODIFIED,
			ROWS_SELECTED as ROWS_RETURNED
		FROM SYSIBMADM.SNAPDB
	`

	queryBufferpoolAggregateMetrics = `
		SELECT 
			-- Raw counter values for proper time-series hit ratio calculation
			-- We need to subtract async pages as per IBM formula
			SUM(POOL_DATA_LBP_PAGES_FOUND - POOL_ASYNC_DATA_LBP_PAGES_FOUND) as DATA_PAGES_FOUND,
			SUM(POOL_INDEX_LBP_PAGES_FOUND - POOL_ASYNC_INDEX_LBP_PAGES_FOUND) as INDEX_PAGES_FOUND,
			SUM(POOL_XDA_LBP_PAGES_FOUND - POOL_ASYNC_XDA_LBP_PAGES_FOUND) as XDA_PAGES_FOUND,
			SUM(POOL_COL_LBP_PAGES_FOUND - POOL_ASYNC_COL_LBP_PAGES_FOUND) as COL_PAGES_FOUND,
			SUM(POOL_DATA_L_READS) as DATA_L_READS,
			SUM(POOL_INDEX_L_READS) as INDEX_L_READS,
			SUM(POOL_XDA_L_READS) as XDA_L_READS,
			SUM(POOL_COL_L_READS) as COL_L_READS,
			SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS) as LOGICAL_READS,
			SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) as PHYSICAL_READS,
			SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS + 
			    POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) as TOTAL_READS,
			-- Detailed read metrics
			SUM(POOL_DATA_L_READS) as DATA_LOGICAL_READS,
			SUM(POOL_DATA_P_READS) as DATA_PHYSICAL_READS,
			SUM(POOL_DATA_L_READS + POOL_DATA_P_READS) as DATA_TOTAL_READS,
			SUM(POOL_INDEX_L_READS) as INDEX_LOGICAL_READS,
			SUM(POOL_INDEX_P_READS) as INDEX_PHYSICAL_READS,
			SUM(POOL_INDEX_L_READS + POOL_INDEX_P_READS) as INDEX_TOTAL_READS,
			SUM(POOL_XDA_L_READS) as XDA_LOGICAL_READS,
			SUM(POOL_XDA_P_READS) as XDA_PHYSICAL_READS,
			SUM(POOL_XDA_L_READS + POOL_XDA_P_READS) as XDA_TOTAL_READS,
			SUM(POOL_COL_L_READS) as COLUMN_LOGICAL_READS,
			SUM(POOL_COL_P_READS) as COLUMN_PHYSICAL_READS,
			SUM(POOL_COL_L_READS + POOL_COL_P_READS) as COLUMN_TOTAL_READS
		FROM SYSIBMADM.SNAPBP
	`

	queryLogSpaceMetrics = `
		SELECT 
			TOTAL_LOG_USED,
			TOTAL_LOG_AVAILABLE,
			CASE 
				WHEN (TOTAL_LOG_USED + TOTAL_LOG_AVAILABLE) > 0 
				THEN (TOTAL_LOG_USED * 100) / (TOTAL_LOG_USED + TOTAL_LOG_AVAILABLE)
				ELSE 0 
			END as LOG_UTILIZATION,
			(SELECT SUM(LOG_READS) FROM SYSIBMADM.SNAPDB) as LOG_READS,
			(SELECT SUM(LOG_WRITES) FROM SYSIBMADM.SNAPDB) as LOG_WRITES
		FROM SYSIBMADM.LOG_UTILIZATION
	`

	queryLongRunningQueries = `
		SELECT 
			COUNT(*) as TOTAL_COUNT,
			SUM(CASE WHEN ELAPSED_TIME_MIN >= 5 AND ELAPSED_TIME_MIN < 15 THEN 1 ELSE 0 END) as WARNING_COUNT,
			SUM(CASE WHEN ELAPSED_TIME_MIN >= 15 THEN 1 ELSE 0 END) as CRITICAL_COUNT
		FROM SYSIBMADM.LONG_RUNNING_SQL
		WHERE ELAPSED_TIME_MIN > 0
	`

	queryBackupStatus = `
		SELECT 
			OPERATION,
			START_TIME,
			OPERATIONTYPE,
			SQLCODE
		FROM SYSIBMADM.DB_HISTORY
		WHERE OPERATION = 'B'
		  AND START_TIME >= CURRENT TIMESTAMP - %d DAYS
		ORDER BY START_TIME DESC
		FETCH FIRST 10 ROWS ONLY
	`
	queryBackupStatusSimple = `
		SELECT 
			MAX(START_TIME) as LAST_BACKUP
		FROM SYSIBMADM.DB_HISTORY
		WHERE OPERATION = 'B' AND SQLCODE = 0
	`

	queryDatabaseInstances = `
		SELECT 
			DB_NAME,
			DB_STATUS,
			APPLS_CUR_CONS
		FROM SYSIBMADM.SNAPDB
	`

	queryBufferpoolInstances = `
		SELECT 
			BP_NAME,
			PAGESIZE,
			NPAGES as TOTAL_PAGES,
			NPAGES - FREE_PAGES as USED_PAGES,
			-- Raw counters for hit/miss calculation
			POOL_DATA_LBP_PAGES_FOUND - POOL_ASYNC_DATA_LBP_PAGES_FOUND as DATA_PAGES_FOUND,
			POOL_INDEX_LBP_PAGES_FOUND - POOL_ASYNC_INDEX_LBP_PAGES_FOUND as INDEX_PAGES_FOUND,
			POOL_XDA_LBP_PAGES_FOUND - POOL_ASYNC_XDA_LBP_PAGES_FOUND as XDA_PAGES_FOUND,
			POOL_COL_LBP_PAGES_FOUND - POOL_ASYNC_COL_LBP_PAGES_FOUND as COL_PAGES_FOUND,
			-- Read metrics
			POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS as LOGICAL_READS,
			POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS as PHYSICAL_READS,
			POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS + 
			POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS as TOTAL_READS,
			POOL_DATA_L_READS as DATA_LOGICAL_READS,
			POOL_DATA_P_READS as DATA_PHYSICAL_READS,
			POOL_INDEX_L_READS as INDEX_LOGICAL_READS,
			POOL_INDEX_P_READS as INDEX_PHYSICAL_READS,
			POOL_XDA_L_READS as XDA_LOGICAL_READS,
			POOL_XDA_P_READS as XDA_PHYSICAL_READS,
			POOL_COL_L_READS as COLUMN_LOGICAL_READS,
			POOL_COL_P_READS as COLUMN_PHYSICAL_READS,
			-- Write metrics
			POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS as WRITES
		FROM SYSIBMADM.SNAPBP
	`

	queryTablespaceInstances = `
		SELECT 
			TBSP_NAME,
			TBSP_TYPE,
			TBSP_CONTENT_TYPE,
			TBSP_STATE,
			TBSP_TOTAL_SIZE_KB * 1024 as TOTAL_SIZE,
			TBSP_USED_SIZE_KB * 1024 as USED_SIZE,
			TBSP_FREE_SIZE_KB * 1024 as FREE_SIZE,
			TBSP_USABLE_SIZE_KB * 1024 as USABLE_SIZE,
			CASE 
				WHEN TBSP_TOTAL_SIZE_KB > 0 
				THEN (TBSP_USED_SIZE_KB * 100) / TBSP_TOTAL_SIZE_KB
				ELSE 0 
			END as USED_PERCENT,
			TBSP_PAGE_SIZE
		FROM SYSIBMADM.TBSP_UTILIZATION
		WHERE TBSP_TYPE = 'DMS'
	`

	queryConnectionInstances = `
		SELECT 
			APPLICATION_ID,
			APPLICATION_NAME,
			CLIENT_HOSTNAME,
			SESSION_AUTH_ID,
			APPL_STATUS,
			UOW_COMP_STATUS,
			ROWS_READ,
			ROWS_WRITTEN,
			TOTAL_CPU_TIME
		FROM SYSIBMADM.APPLICATIONS
		WHERE APPL_STATUS IN ('CONNECTED', 'UOWEXEC')
	`

	// Table metrics using MON_GET_TABLE
	queryTableInstances = `
		SELECT 
			T.TABSCHEMA,
			T.TABNAME,
			T.DATA_OBJECT_L_PAGES as DATA_OBJECT_P_SIZE,
			T.INDEX_OBJECT_L_PAGES as INDEX_OBJECT_P_SIZE,
			T.LOB_OBJECT_L_PAGES as LONG_OBJECT_P_SIZE,
			T.ROWS_READ,
			T.ROWS_INSERTED + T.ROWS_UPDATED + T.ROWS_DELETED as ROWS_WRITTEN
		FROM TABLE(MON_GET_TABLE('','',-2)) AS T
		WHERE T.TABSCHEMA NOT LIKE 'SYS%'
	`

	// Index metrics using MON_GET_INDEX
	queryIndexInstances = `
		SELECT
			I.TABSCHEMA as INDSCHEMA,
			CAST(I.IID AS VARCHAR(128)) as INDNAME,
			I.NLEAF,
			I.INDEX_SCANS,
			I.INDEX_ONLY_SCANS as FULL_SCANS
		FROM TABLE(MON_GET_INDEX('','',-2)) AS I
		WHERE I.TABSCHEMA NOT LIKE 'SYS%'
	`

	queryDatabaseStatus = `
		SELECT 
			CASE 
				WHEN EXISTS (SELECT 1 FROM SYSIBMADM.SNAPDB WHERE DB_STATUS = 'ACTIVE') THEN 0
				WHEN EXISTS (SELECT 1 FROM SYSIBMADM.SNAPDB WHERE DB_STATUS = 'QUIESCED') THEN 2
				WHEN EXISTS (SELECT 1 FROM SYSIBMADM.SNAPDB WHERE DB_STATUS IN ('QUIESCE_PENDING', 'ROLLFORWARD')) THEN 1
				ELSE 3
			END as DB_STATUS_CHECK
		FROM SYSIBM.SYSDUMMY1
	`

	queryCanConnect = `SELECT 1 as CAN_CONNECT FROM SYSIBM.SYSDUMMY1`

	// Individual metric queries for resilience (following AS/400 pattern)
	// Core connection metrics - should always be available on LUW
	queryTotalConnections     = `SELECT COUNT(*) FROM SYSIBMADM.APPLICATIONS`
	queryActiveConnections    = `SELECT COUNT(*) FROM SYSIBMADM.APPLICATIONS WHERE APPL_STATUS = 'CONNECTED'`
	queryExecutingConnections = `SELECT COUNT(*) FROM SYSIBMADM.APPLICATIONS WHERE APPL_STATUS = 'UOWEXEC'`
	queryIdleConnections      = `SELECT COUNT(*) FROM SYSIBMADM.APPLICATIONS WHERE APPL_STATUS = 'CONNECTED' AND UOW_COMP_STATUS = 'NONE'`
	queryMaxConnections       = `SELECT COALESCE(NULLIF(VALUE, 'AUTOMATIC'), '40') FROM SYSIBMADM.DBCFG WHERE NAME = 'maxappls'`

	// Core lock metrics - should be available on most DB2 editions
	queryLockWaits         = `SELECT SUM(LOCK_WAITS) FROM SYSIBMADM.SNAPDB`
	queryLockTimeouts      = `SELECT SUM(LOCK_TIMEOUTS) FROM SYSIBMADM.SNAPDB`
	queryDeadlocks         = `SELECT SUM(DEADLOCKS) FROM SYSIBMADM.SNAPDB`
	queryLockEscalations   = `SELECT SUM(LOCK_ESCALS) FROM SYSIBMADM.SNAPDB`
	queryActiveLocks       = `SELECT SUM(LOCKS_HELD) FROM SYSIBMADM.SNAPDB`
	queryLockWaitTime      = `SELECT SUM(LOCK_WAIT_TIME) FROM SYSIBMADM.SNAPDB`
	queryLockWaitingAgents = `SELECT SUM(LOCKS_WAITING) FROM SYSIBMADM.SNAPDB`
	queryLockMemoryPages   = `SELECT SUM(LOCK_LIST_IN_USE) FROM SYSIBMADM.SNAPDB`

	// Sorting metrics
	queryTotalSorts    = `SELECT SUM(TOTAL_SORTS) FROM SYSIBMADM.SNAPDB`
	querySortOverflows = `SELECT SUM(SORT_OVERFLOWS) FROM SYSIBMADM.SNAPDB`

	// Row activity metrics
	queryRowsRead     = `SELECT SUM(ROWS_READ) FROM SYSIBMADM.SNAPDB`
	queryRowsModified = `SELECT SUM(ROWS_MODIFIED) FROM SYSIBMADM.SNAPDB`
	queryRowsReturned = `SELECT SUM(ROWS_SELECTED) FROM SYSIBMADM.SNAPDB`

	// Log space metrics
	queryLogUsedSpace      = `SELECT SUM(TOTAL_LOG_USED) FROM SYSIBMADM.LOG_UTILIZATION`
	queryLogAvailableSpace = `SELECT SUM(TOTAL_LOG_AVAILABLE) FROM SYSIBMADM.LOG_UTILIZATION`
	queryLogReads          = `SELECT SUM(LOG_READS) FROM SYSIBMADM.SNAPDB`
	queryLogWrites         = `SELECT SUM(LOG_WRITES) FROM SYSIBMADM.SNAPDB`

	// Buffer pool aggregate metrics
	queryBufferpoolLogicalReads   = `SELECT SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolPhysicalReads  = `SELECT SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolDataLogical    = `SELECT SUM(POOL_DATA_L_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolDataPhysical   = `SELECT SUM(POOL_DATA_P_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolIndexLogical   = `SELECT SUM(POOL_INDEX_L_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolIndexPhysical  = `SELECT SUM(POOL_INDEX_P_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolXDALogical     = `SELECT SUM(POOL_XDA_L_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolXDAPhysical    = `SELECT SUM(POOL_XDA_P_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolColumnLogical  = `SELECT SUM(POOL_COL_L_READS) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolColumnPhysical = `SELECT SUM(POOL_COL_P_READS) FROM SYSIBMADM.SNAPBP`

	// Hit ratio components for calculation
	queryBufferpoolDataHits   = `SELECT SUM(POOL_DATA_LBP_PAGES_FOUND) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolIndexHits  = `SELECT SUM(POOL_INDEX_LBP_PAGES_FOUND) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolXDAHits    = `SELECT SUM(POOL_XDA_LBP_PAGES_FOUND) FROM SYSIBMADM.SNAPBP`
	queryBufferpoolColumnHits = `SELECT SUM(POOL_COL_LBP_PAGES_FOUND) FROM SYSIBMADM.SNAPBP`

	// Long running queries
	queryLongRunningTotal    = `SELECT COUNT(*) FROM SYSIBMADM.LONG_RUNNING_SQL WHERE ELAPSED_TIME_MIN > 0`
	queryLongRunningWarning  = `SELECT COUNT(*) FROM SYSIBMADM.LONG_RUNNING_SQL WHERE ELAPSED_TIME_MIN >= 5 AND ELAPSED_TIME_MIN < 15`
	queryLongRunningCritical = `SELECT COUNT(*) FROM SYSIBMADM.LONG_RUNNING_SQL WHERE ELAPSED_TIME_MIN >= 15`

	// Modern MON_GET_* function queries (DB2 9.7+ LUW)
	// These provide better performance and real-time data compared to SNAP* views

	// Database-level metrics using MON_GET_DATABASE
	queryMonGetDatabase = `
		SELECT 
			SUM(LOCK_WAITS) as LOCK_WAITS,
			SUM(LOCK_TIMEOUTS) as LOCK_TIMEOUTS,
			SUM(DEADLOCKS) as DEADLOCKS,
			SUM(LOCK_ESCALS) as LOCK_ESCALS,
			SUM(NUM_LOCKS_HELD) as LOCK_ACTIVE,
			SUM(LOCK_WAIT_TIME) as LOCK_WAIT_TIME,
			SUM(NUM_LOCKS_WAITING) as LOCK_WAITING_AGENTS,
			SUM(LOCK_LIST_IN_USE) as LOCK_MEMORY_PAGES,
			SUM(TOTAL_SORTS) as TOTAL_SORTS,
			SUM(SORT_OVERFLOWS) as SORT_OVERFLOWS,
			SUM(ROWS_READ) as ROWS_READ,
			SUM(ROWS_MODIFIED) as ROWS_MODIFIED,
			SUM(ROWS_RETURNED) as ROWS_RETURNED
		FROM TABLE(MON_GET_DATABASE(-2)) AS T
	`

	// Buffer pool metrics using MON_GET_BUFFERPOOL
	queryMonGetBufferpool = `
		SELECT 
			BP_NAME,
			POOL_DATA_L_READS,
			POOL_DATA_P_READS,
			POOL_INDEX_L_READS,
			POOL_INDEX_P_READS,
			POOL_XDA_L_READS,
			POOL_XDA_P_READS,
			POOL_COL_L_READS,
			POOL_COL_P_READS,
			POOL_DATA_LBP_PAGES_FOUND,
			POOL_INDEX_LBP_PAGES_FOUND,
			POOL_XDA_LBP_PAGES_FOUND,
			POOL_COL_LBP_PAGES_FOUND,
			AUTOMATIC,
			BP_CUR_BUFFSZ as POOL_CUR_SIZE,
			0 as POOL_WATERMARK
		FROM TABLE(MON_GET_BUFFERPOOL('',-2)) AS T
	`

	// Aggregate buffer pool metrics - use MON_GET approach (hits = logical - physical)
	queryMonGetBufferpoolAggregate = `
		SELECT 
			SUM(POOL_DATA_L_READS) as DATA_LOGICAL_READS,
			SUM(POOL_DATA_P_READS) as DATA_PHYSICAL_READS,
			SUM(POOL_INDEX_L_READS) as INDEX_LOGICAL_READS,
			SUM(POOL_INDEX_P_READS) as INDEX_PHYSICAL_READS,
			SUM(POOL_XDA_L_READS) as XDA_LOGICAL_READS,
			SUM(POOL_XDA_P_READS) as XDA_PHYSICAL_READS,
			SUM(POOL_COL_L_READS) as COLUMN_LOGICAL_READS,
			SUM(POOL_COL_P_READS) as COLUMN_PHYSICAL_READS,
			SUM(POOL_DATA_L_READS - POOL_DATA_P_READS) as DATA_HITS,
			SUM(POOL_INDEX_L_READS - POOL_INDEX_P_READS) as INDEX_HITS,
			SUM(POOL_XDA_L_READS - POOL_XDA_P_READS) as XDA_HITS,
			SUM(POOL_COL_L_READS - POOL_COL_P_READS) as COLUMN_HITS,
			SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES) as TOTAL_WRITES
		FROM TABLE(MON_GET_BUFFERPOOL('',-2)) AS T
	`

	// Transaction log metrics using MON_GET_TRANSACTION_LOG
	queryMonGetTransactionLog = `
		SELECT 
			TOTAL_LOG_USED as TOTAL_LOG_USED,
			TOTAL_LOG_AVAILABLE as TOTAL_LOG_AVAILABLE,
			CASE 
				WHEN TOTAL_LOG_AVAILABLE > 0 
				THEN (TOTAL_LOG_USED * 100) / TOTAL_LOG_AVAILABLE
				ELSE 0 
			END as LOG_UTILIZATION,
			LOG_READS as LOG_READS,
			LOG_WRITES as LOG_WRITES
		FROM TABLE(MON_GET_TRANSACTION_LOG(-2)) AS T
		WHERE MEMBER = 0
	`

	// Tablespace metrics using MON_GET_TABLESPACE
	queryMonGetTablespace = `
		SELECT 
			TBSP_NAME,
			TBSP_TYPE,
			TBSP_CONTENT_TYPE,
			TBSP_STATE,
			TBSP_TOTAL_PAGES * TBSP_PAGE_SIZE as TOTAL_SIZE,
			TBSP_USED_PAGES * TBSP_PAGE_SIZE as USED_SIZE,
			TBSP_FREE_PAGES * TBSP_PAGE_SIZE as FREE_SIZE,
			(TBSP_TOTAL_PAGES - TBSP_PENDING_FREE_PAGES) * TBSP_PAGE_SIZE as USABLE_SIZE,
			CASE 
				WHEN TBSP_TOTAL_PAGES > 0 
				THEN (TBSP_USED_PAGES * 100) / TBSP_TOTAL_PAGES
				ELSE 0 
			END as USED_PERCENT,
			TBSP_PAGE_SIZE
		FROM TABLE(MON_GET_TABLESPACE('',-2)) AS T
		WHERE TBSP_TYPE = 'DMS'
	`

	// Connection metrics using MON_GET_CONNECTION
	queryMonGetConnections = `
		SELECT 
			COUNT(*) as TOTAL_CONNS,
			SUM(CASE WHEN APPLICATION_HANDLE IS NOT NULL THEN 1 ELSE 0 END) as ACTIVE_CONNS,
			SUM(CASE WHEN CLIENT_IDLE_WAIT_TIME > 0 THEN 1 ELSE 0 END) as IDLE_CONNS,
			SUM(CASE WHEN UOW_START_TIME IS NOT NULL THEN 1 ELSE 0 END) as EXECUTING_CONNS
		FROM TABLE(MON_GET_CONNECTION(NULL,-2)) AS T
	`

	// Individual connection details
	// FIXED: Removed dynamic "top X" ORDER BY + FETCH FIRST pattern for stable time-series
	queryMonGetConnectionDetails = `
		SELECT 
			APPLICATION_ID,
			APPLICATION_NAME,
			CLIENT_HOSTNAME,
			CLIENT_IPADDR,
			SESSION_AUTH_ID,
			CASE 
				WHEN UOW_START_TIME IS NOT NULL THEN 'UOWEXEC'
				ELSE 'CONNECTED'
			END as APPL_STATUS,
			ROWS_READ,
			ROWS_MODIFIED + ROWS_INSERTED + ROWS_UPDATED + ROWS_DELETED as ROWS_WRITTEN,
			TOTAL_CPU_TIME
		FROM TABLE(MON_GET_CONNECTION(NULL,-2)) AS T
		-- WHERE APPLICATION_HANDLE IS NOT NULL
	`

	// Memory pool monitoring using MON_GET_MEMORY_POOL
	queryMonGetMemoryPool = `
		SELECT 
			MEMORY_SET_TYPE,
			MEMORY_POOL_TYPE,
			MEMORY_POOL_USED,
			MEMORY_POOL_USED_HWM
		FROM TABLE(MON_GET_MEMORY_POOL('DATABASE', '', -2)) AS T
		ORDER BY MEMORY_POOL_USED DESC
	`

	// Instance Memory Sets monitoring (Screen 26) using MON_GET_MEMORY_SET
	// FIXED: Removed dynamic "top X" ORDER BY + FETCH FIRST pattern for stable time-series
	queryMonGetMemorySet = `
		SELECT 
			HOST_NAME,
			DB_NAME,
			MEMORY_SET_TYPE,
			MEMBER,
			MEMORY_SET_USED,
			MEMORY_SET_COMMITTED,
			MEMORY_SET_USED_HWM,
			(MEMORY_SET_COMMITTED - MEMORY_SET_USED) as ADDITIONAL_COMMITTED,
			CASE 
				WHEN MEMORY_SET_USED_HWM > 0 
				THEN (MEMORY_SET_USED * 100) / MEMORY_SET_USED_HWM
				ELSE 0 
			END as PERCENT_USED_HWM
		FROM TABLE(MON_GET_MEMORY_SET(NULL, NULL, -2)) AS T
	`

	// Enhanced connection wait metrics
	// FIXED: Removed dynamic "top X" ORDER BY + FETCH FIRST pattern for stable time-series
	queryMonGetConnectionWaits = `
		SELECT 
			APPLICATION_ID,
			TOTAL_WAIT_TIME,
			LOCK_WAIT_TIME,
			DIRECT_READ_TIME + DIRECT_WRITE_TIME as IO_WAIT_TIME,
			CLIENT_IDLE_WAIT_TIME,
			TOTAL_RQST_TIME,
			TOTAL_ACT_TIME
		FROM TABLE(MON_GET_CONNECTION(NULL, -2)) AS T
		WHERE APPLICATION_HANDLE IS NOT NULL
			AND TOTAL_WAIT_TIME > 0
	`

	// Table I/O statistics using MON_GET_TABLE
	// FIXED: Removed dynamic "top X" ORDER BY + FETCH FIRST pattern for stable time-series
	queryMonGetTable = `
		SELECT 
			TABSCHEMA,
			TABNAME,
			TABLE_SCANS,
			ROWS_READ,
			ROWS_INSERTED,
			ROWS_UPDATED,
			ROWS_DELETED,
			OVERFLOW_ACCESSES
		FROM TABLE(MON_GET_TABLE(NULL, NULL, -2)) AS T
	`

	// Enhanced buffer pool metrics
	queryMonGetBufferpoolEnhanced = `
		SELECT 
			BP_NAME,
			POOL_DATA_L_READS,
			POOL_DATA_P_READS,
			POOL_INDEX_L_READS,
			POOL_INDEX_P_READS,
			POOL_XDA_L_READS,
			POOL_XDA_P_READS,
			POOL_COL_L_READS,
			POOL_COL_P_READS,
			POOL_DATA_LBP_PAGES_FOUND,
			POOL_INDEX_LBP_PAGES_FOUND,
			POOL_XDA_LBP_PAGES_FOUND,
			POOL_COL_LBP_PAGES_FOUND,
			POOL_ASYNC_DATA_READS,
			POOL_ASYNC_DATA_WRITES,
			POOL_ASYNC_INDEX_READS,
			POOL_ASYNC_INDEX_WRITES,
			POOL_ASYNC_XDA_READS,
			POOL_ASYNC_XDA_WRITES,
			PREFETCH_WAIT_TIME,
			POOL_DRTY_PG_STEAL_CLNS,
			POOL_DRTY_PG_THRSH_CLNS,
			AUTOMATIC,
			BP_CUR_BUFFSZ as POOL_CUR_SIZE,
			0 as POOL_WATERMARK
		FROM TABLE(MON_GET_BUFFERPOOL('',-2)) AS T
	`

	// Database Overview queries (Screen 01)
	queryDatabaseOverview = `
		WITH CPU_METRICS AS (
			SELECT 
				CPU_USER * 100 AS CPU_USER_PERCENT,
				CPU_SYSTEM * 100 AS CPU_SYSTEM_PERCENT,
				CPU_IDLE * 100 AS CPU_IDLE_PERCENT,
				CPU_IOWAIT * 100 AS CPU_IOWAIT_PERCENT
			FROM TABLE(SYSPROC.ENV_GET_SYSTEM_RESOURCES()) AS T
		),
		CONNECTION_METRICS AS (
			SELECT 
				SUM(CASE WHEN APPLICATION_HANDLE IS NOT NULL THEN 1 ELSE 0 END) AS CONNECTIONS_ACTIVE,
				COUNT(*) AS CONNECTIONS_TOTAL
			FROM TABLE(MON_GET_CONNECTION(NULL, -2)) AS T
		),
		MEMORY_METRICS AS (
			SELECT 
				SUM(CASE WHEN MEMORY_SET_TYPE = 'DATABASE' THEN MEMORY_SET_COMMITTED ELSE 0 END) AS MEMORY_DATABASE_COMMITTED,
				SUM(CASE WHEN MEMORY_SET_TYPE = 'INSTANCE' THEN MEMORY_SET_COMMITTED ELSE 0 END) AS MEMORY_INSTANCE_COMMITTED,
				(SELECT SUM(MEMORY_POOL_USED) FROM TABLE(MON_GET_MEMORY_POOL('DATABASE', '', -2)) WHERE MEMORY_POOL_TYPE = 'BP') AS MEMORY_BUFFERPOOL_USED,
				(SELECT SUM(MEMORY_POOL_USED) FROM TABLE(MON_GET_MEMORY_POOL('DATABASE', '', -2)) WHERE MEMORY_POOL_TYPE = 'SHARED_SORT') AS MEMORY_SHARED_SORT_USED
			FROM TABLE(MON_GET_MEMORY_SET(NULL, NULL, -2))
		),
		THROUGHPUT_METRICS AS (
			SELECT 
				SUM(SELECT_SQL_STMTS) AS OPS_SELECT_STMTS,
				SUM(UID_SQL_STMTS) AS OPS_UID_STMTS,
				SUM(TOTAL_APP_COMMITS + TOTAL_APP_ROLLBACKS) AS OPS_TRANSACTIONS,
				0 AS OPS_ACTIVITIES_ABORTED
			FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
		),
		TIME_SPENT AS (
			SELECT 
				CASE WHEN SUM(DIRECT_READS) > 0 
					THEN SUM(DIRECT_READ_TIME) / SUM(DIRECT_READS) ELSE 0 END AS TIME_AVG_DIRECT_READ,
				CASE WHEN SUM(DIRECT_WRITES) > 0 
					THEN SUM(DIRECT_WRITE_TIME) / SUM(DIRECT_WRITES) ELSE 0 END AS TIME_AVG_DIRECT_WRITE,
				CASE WHEN SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) > 0 
					THEN SUM(POOL_READ_TIME) / SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) ELSE 0 END AS TIME_AVG_POOL_READ,
				CASE WHEN SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) > 0 
					THEN SUM(POOL_WRITE_TIME) / SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) ELSE 0 END AS TIME_AVG_POOL_WRITE
			FROM TABLE(MON_GET_BUFFERPOOL('', -2))
		),
		DB_STATUS AS (
			SELECT 
				CASE WHEN DB_STATUS = 'ACTIVE' THEN 1 ELSE 0 END AS DATABASE_ACTIVE,
				CASE WHEN DB_STATUS != 'ACTIVE' THEN 1 ELSE 0 END AS DATABASE_INACTIVE
			FROM TABLE(MON_GET_DATABASE(-2))
			FETCH FIRST 1 ROW ONLY
		)
		SELECT * FROM CPU_METRICS, CONNECTION_METRICS, MEMORY_METRICS, 
					  THROUGHPUT_METRICS, TIME_SPENT, DB_STATUS
	`

	// Alternative simpler query for older DB2 versions using SNAP views
	queryDatabaseOverviewSimple = `
		SELECT 
			-- Database status (always active when we can query)
			1 AS DATABASE_ACTIVE,
			0 AS DATABASE_INACTIVE,
			-- CPU metrics (not available in SNAP views)
			0 AS CPU_USER_PERCENT,
			0 AS CPU_SYSTEM_PERCENT,
			0 AS CPU_IDLE_PERCENT,
			0 AS CPU_IOWAIT_PERCENT,
			-- Connection metrics from SNAPAPPL_INFO
			(SELECT COUNT(*) FROM SYSIBMADM.SNAPAPPL_INFO WHERE AGENT_ID > 0) AS CONNECTIONS_ACTIVE,
			(SELECT COUNT(*) FROM SYSIBMADM.SNAPAPPL_INFO) AS CONNECTIONS_TOTAL,
			-- Memory metrics (basic, from SNAPDB)
			COALESCE((SELECT SUM(POOL_CUR_SIZE) FROM SYSIBMADM.SNAPDB), 0) AS MEMORY_DATABASE_COMMITTED,
			0 AS MEMORY_INSTANCE_COMMITTED,
			0 AS MEMORY_BUFFERPOOL_USED,
			0 AS MEMORY_SHARED_SORT_USED,
			-- Throughput metrics from SNAPDB
			COALESCE((SELECT SUM(SELECT_SQL_STMTS) FROM SYSIBMADM.SNAPDB), 0) AS OPS_SELECT_STMTS,
			COALESCE((SELECT SUM(UID_SQL_STMTS) FROM SYSIBMADM.SNAPDB), 0) AS OPS_UID_STMTS,
			COALESCE((SELECT SUM(TOTAL_APP_COMMITS + INT_COMMITS + TOTAL_APP_ROLLBACKS + INT_ROLLBACKS) FROM SYSIBMADM.SNAPDB), 0) AS OPS_TRANSACTIONS,
			0 AS OPS_ACTIVITIES_ABORTED,
			-- Time spent (basic averages from SNAPDB)
			COALESCE((SELECT CASE WHEN SUM(DIRECT_READS) > 0 THEN SUM(DIRECT_READ_TIME) / SUM(DIRECT_READS) ELSE 0 END FROM SYSIBMADM.SNAPDB), 0) AS TIME_AVG_DIRECT_READ,
			COALESCE((SELECT CASE WHEN SUM(DIRECT_WRITES) > 0 THEN SUM(DIRECT_WRITE_TIME) / SUM(DIRECT_WRITES) ELSE 0 END FROM SYSIBMADM.SNAPDB), 0) AS TIME_AVG_DIRECT_WRITE,
			COALESCE((SELECT CASE WHEN SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) > 0 THEN SUM(POOL_READ_TIME) / SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) ELSE 0 END FROM SYSIBMADM.SNAPBP), 0) AS TIME_AVG_POOL_READ,
			COALESCE((SELECT CASE WHEN SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) > 0 THEN SUM(POOL_WRITE_TIME) / SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) ELSE 0 END FROM SYSIBMADM.SNAPBP), 0) AS TIME_AVG_POOL_WRITE
		FROM SYSIBM.SYSDUMMY1
	`

	// Enhanced logging performance query (Screen 18) - using tested dcmtop queries
	queryLoggingPerformance = `
		SELECT 
			SUM(TOTAL_APP_COMMITS) AS LOG_COMMITS,
			SUM(TOTAL_APP_ROLLBACKS) AS LOG_ROLLBACKS
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	queryLoggingReads = `
		SELECT SUM(NUM_LOG_READ_IO) AS LOG_READS FROM TABLE(MON_GET_TRANSACTION_LOG(-2))
	`

	queryLoggingWrites = `
		SELECT SUM(NUM_LOG_WRITE_IO) AS LOG_WRITES FROM TABLE(MON_GET_TRANSACTION_LOG(-2))
	`

	queryLoggingBufferEvents = `
		SELECT SUM(NUM_LOG_BUFFER_FULL) AS LOG_BUFFER_FULL_EVENTS FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	// Individual logging queries from dcmtop (Screen 18)
	queryLogCommits = `
		SELECT SUM(TOTAL_APP_COMMITS) AS LOG_COMMITS FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	queryLogRollbacks = `
		SELECT SUM(TOTAL_APP_ROLLBACKS) AS LOG_ROLLBACKS FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	// Simpler version for older DB2
	queryLoggingPerformanceSimple = `
		SELECT 
			SUM(TOTAL_APP_COMMITS) AS LOG_COMMITS,
			SUM(TOTAL_APP_ROLLBACKS) AS LOG_ROLLBACKS,
			SUM(LOG_READS) AS LOG_READS,
			SUM(LOG_WRITES) AS LOG_WRITES,
			0 AS LOG_BUFFER_FULL_EVENTS,
			0 AS LOG_AVG_COMMIT_TIME,
			0 AS LOG_AVG_READ_TIME,
			0 AS LOG_AVG_WRITE_TIME
		FROM SYSIBMADM.SNAPDB
	`

	// Federation metrics query (Screen 32)
	queryFederationMetrics = `
		SELECT 
			SUM(CASE WHEN ROWS_RETURNED_FROM_REMOTE > 0 THEN 1 ELSE 0 END) AS FED_CONNECTIONS_ACTIVE,
			SUM(CASE WHEN ROWS_RETURNED_FROM_REMOTE = 0 THEN 1 ELSE 0 END) AS FED_CONNECTIONS_IDLE,
			SUM(ROWS_RETURNED_FROM_REMOTE) AS FED_ROWS_READ,
			SUM(SELECT_SQL_STMTS) AS FED_SELECT_STMTS,
			SUM(TOTAL_WAIT_TIME) AS FED_WAITS_TOTAL
		FROM TABLE(MON_GET_CONNECTION(NULL, -2))
		WHERE APPLICATION_NAME IS NOT NULL
	`

	// Simpler federation query without FILTER clause
	queryFederationMetricsSimple = `
		SELECT 
			SUM(CASE WHEN ROWS_RETURNED_FROM_REMOTE > 0 THEN 1 ELSE 0 END) AS FED_CONNECTIONS_ACTIVE,
			SUM(CASE WHEN ROWS_RETURNED_FROM_REMOTE = 0 THEN 1 ELSE 0 END) AS FED_CONNECTIONS_IDLE,
			SUM(ROWS_RETURNED_FROM_REMOTE) AS FED_ROWS_READ,
			SUM(SELECT_SQL_STMTS) AS FED_SELECT_STMTS,
			SUM(TOTAL_WAIT_TIME) AS FED_WAITS_TOTAL
		FROM TABLE(MON_GET_CONNECTION(NULL, -2))
		WHERE APPLICATION_NAME IS NOT NULL
	`

	// Memory Sets query (Screen 26)
	queryMemorySets = `
		SELECT 
			HOST_NAME,
			DB_NAME, 
			MEMORY_SET_TYPE,
			MEMBER,
			MEMORY_SET_USED,
			MEMORY_SET_COMMITTED,
			(MEMORY_SET_COMMITTED - MEMORY_SET_USED) AS ADDITIONAL_COMMITTED,
			MEMORY_SET_USED_HWM,
			CASE 
				WHEN SUM(MEMORY_SET_COMMITTED) OVER () > 0 
				THEN (MEMORY_SET_COMMITTED * 100.0) / SUM(MEMORY_SET_COMMITTED) OVER () 
				ELSE 0 
			END AS PERCENT_COMMITTED
		FROM TABLE(MON_GET_MEMORY_SET(NULL, NULL, -2))
		ORDER BY MEMORY_SET_TYPE
	`

	// 	// Prefetcher metrics query (Screen 15)
	// 	queryPrefetcherMetrics = `
	// 		SELECT
	// 			BP_NAME,
	// 			CASE
	// 				WHEN (POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) > 0
	// 				THEN DECIMAL((POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS +
	// 							 POOL_ASYNC_XDA_READS + POOL_ASYNC_COL_READS) * 100.0 /
	// 							(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS), 5, 2)
	// 				ELSE 0.00
	// 			END AS PREFETCH_RATIO,
	// 			CASE
	// 				WHEN (POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES + POOL_COL_WRITES) > 0
	// 				THEN DECIMAL((POOL_ASYNC_DATA_WRITES + POOL_ASYNC_INDEX_WRITES +
	// 							 POOL_ASYNC_XDA_WRITES + POOL_ASYNC_COL_WRITES) * 100.0 /
	// 							(POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES + POOL_COL_WRITES), 5, 2)
	// 				ELSE 0.00
	// 			END AS CLEANER_RATIO,
	// 			(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) AS PHYSICAL_READS,
	// 			(POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS + POOL_ASYNC_XDA_READS + POOL_ASYNC_COL_READS) AS ASYNC_READS,
	// 			UNREAD_PREFETCH_PAGES,
	// 			CASE
	// 				WHEN PREFETCH_WAITS > 0
	// 				THEN DECIMAL(PREFETCH_WAIT_TIME / PREFETCH_WAITS, 10, 2)
	// 				ELSE 0.00
	// 			END AS AVG_WAIT_TIME
	// 		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	// 	`
	//
	// Database Overview Simple Queries (based on dcmtop)
	// These are individual queries that are more resilient and easier to debug

	// Overview queries
	querySimpleDatabaseStatus = `SELECT MAX(DB_STATUS) AS DATABASE_STATUS FROM TABLE(MON_GET_DATABASE(-2))`
	querySimpleStartTime      = `SELECT MIN(DB_CONN_TIME) AS START_TIME FROM TABLE(MON_GET_DATABASE(-2))`
	querySimpleSystemMemory   = `SELECT MEMORY_TOTAL AS SYSTEM_PHYSICAL_MEM FROM TABLE(SYSPROC.ENV_GET_SYSTEM_RESOURCES())`

	// CPU queries
	querySimpleCPUDB2 = `
		SELECT 
			SUM(CPU_USER) AS DB2_CPU_USER_TOTAL,
			SUM(CPU_SYSTEM) AS DB2_CPU_SYSTEM_TOTAL
		FROM TABLE(ENV_GET_DB2_SYSTEM_RESOURCES(-2))
	`
	querySimpleCPUSystem = `
		SELECT 
			CPU_USER AS CPU_USER_TOTAL,
			CPU_SYSTEM AS CPU_SYSTEM_TOTAL,
			CPU_IDLE AS CPU_IDLE_TOTAL,
			CPU_IOWAIT AS CPU_IOWAIT_TOTAL,
			CASE 
				WHEN (CPU_USER + CPU_SYSTEM + CPU_IDLE + CPU_IOWAIT) > 0 
				THEN ((CPU_USER + CPU_SYSTEM) * 100.0) / (CPU_USER + CPU_SYSTEM + CPU_IDLE + CPU_IOWAIT)
				ELSE 0
			END AS SYSTEM_CPU_UTILIZATION
		FROM TABLE(SYSPROC.ENV_GET_SYSTEM_RESOURCES())
	`

	// Memory queries
	querySimpleMemoryInstance = `
		SELECT SUM(MEMORY_SET_COMMITTED) AS INSTANCE_MEM_COMMITTED
		FROM TABLE(MON_GET_MEMORY_SET('DBMS', CURRENT_SERVER, -2))
	`
	querySimpleMemoryDatabase = `
		SELECT SUM(MEMORY_SET_COMMITTED) AS DATABASE_MEM_COMMITTED
		FROM TABLE(MON_GET_MEMORY_SET('DATABASE', CURRENT_SERVER, -2))
	`
	querySimpleMemoryBufferpool = `
		SELECT SUM(MEMORY_POOL_USED) AS BUFFERPOOL_MEM_USED
		FROM TABLE(MON_GET_MEMORY_POOL('DATABASE', '', -2))
		WHERE MEMORY_POOL_TYPE = 'BP'
	`
	querySimpleMemorySharedSort = `
		SELECT SUM(MEMORY_POOL_USED) AS SHARED_SORT_MEM_USED
		FROM TABLE(MON_GET_MEMORY_POOL('DATABASE', '', -2))
		WHERE MEMORY_POOL_TYPE = 'SHARED_SORT'
	`

	// Storage query
	querySimpleStorageUsage = `
		SELECT 
			SUM(TBSP_TOTAL_PAGES * TBSP_PAGE_SIZE) AS TOTAL_STORAGE,
			SUM(TBSP_USED_PAGES * TBSP_PAGE_SIZE) AS USED_STORAGE,
			SUM(TBSP_FREE_PAGES * TBSP_PAGE_SIZE) AS FREE_STORAGE
		FROM TABLE(MON_GET_TABLESPACE('', -2))
		WHERE TBSP_TYPE = 'DMS'
	`

	// Throughput queries
	querySimpleTransactions = `
		SELECT SUM(TOTAL_APP_COMMITS + INT_COMMITS + TOTAL_APP_ROLLBACKS + INT_ROLLBACKS) AS TRANSACTIONS
		FROM TABLE(MON_GET_DATABASE(NULL))
	`
	querySimpleSelectStmts = `
		SELECT SUM(SELECT_SQL_STMTS) AS SELECT_STMTS
		FROM TABLE(MON_GET_DATABASE(NULL))
	`
	querySimpleUIDStmts = `
		SELECT SUM(UID_SQL_STMTS) AS UID_STMTS
		FROM TABLE(MON_GET_DATABASE(NULL))
	`
	querySimpleActivitiesAborted = `
		SELECT SUM(ACT_ABORTED_TOTAL) AS ACTIVITIES_ABORTED
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, NULL))
	`
	querySnapActivitiesAborted  = `SELECT 0 AS ACTIVITIES_ABORTED`
	querySimpleActivitiesQueued = `
		SELECT SUM(WLM_QUEUE_ASSIGNMENTS_TOTAL) AS ACTIVITIES_QUEUED
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`
	querySimpleLogReads = `
		SELECT SUM(NUM_LOG_READ_IO) AS LOG_READS
		FROM TABLE(MON_GET_TRANSACTION_LOG(-2))
	`
	querySimpleLogWrites = `
		SELECT SUM(NUM_LOG_WRITE_IO) AS LOG_WRITES
		FROM TABLE(MON_GET_TRANSACTION_LOG(-2))
	`
	querySimpleLogicalReads = `
		SELECT SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + POOL_COL_L_READS) AS LOGICAL_READS
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimplePhysicalReads = `
		SELECT SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS) AS PHYSICAL_READS
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimpleAsyncReads = `
		SELECT SUM(POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS + POOL_ASYNC_XDA_READS + POOL_ASYNC_COL_READS) AS ASYNC_READS
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimpleWrites = `
		SELECT SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES + POOL_COL_WRITES) AS WRITES
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimpleAsyncWrites = `
		SELECT SUM(POOL_ASYNC_DATA_WRITES + POOL_ASYNC_INDEX_WRITES + POOL_ASYNC_XDA_WRITES + POOL_ASYNC_COL_WRITES) AS ASYNC_WRITES
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`

	// Contention queries
	querySimpleConnectionsActive = `
		SELECT COUNT(*) AS ACTIVE_CONNECTIONS
		FROM TABLE(MON_GET_CONNECTION(NULL, -2))
		WHERE APPLICATION_HANDLE IS NOT NULL
	`
	querySimpleConnectionsTotal = `
		SELECT COUNT(*) AS TOTAL_CONNECTIONS
		FROM TABLE(MON_GET_CONNECTION(NULL, -2))
	`
	querySimpleLockHeld = `
		SELECT SUM(NUM_LOCKS_HELD) AS LOCKS_HELD
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleLockWaits = `
		SELECT SUM(LOCK_WAITS) AS LOCK_WAITS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleLockTimeouts = `
		SELECT SUM(LOCK_TIMEOUTS) AS LOCK_TIMEOUTS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleLockEscalations = `
		SELECT SUM(LOCK_ESCALS) AS LOCK_ESCALATIONS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleDeadlocks = `
		SELECT SUM(DEADLOCKS) AS DEADLOCKS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleThresholdViolations = `
		SELECT SUM(THRESH_VIOLATIONS) AS THRESHOLD_VIOLATIONS
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`
	querySimpleHitRatio = `
		SELECT 
			SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + 
				POOL_TEMP_DATA_L_READS + POOL_TEMP_INDEX_L_READS + POOL_TEMP_XDA_L_READS + 
				POOL_COL_L_READS + POOL_TEMP_COL_L_READS) AS LOGICAL_READS,
			SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + 
				POOL_TEMP_DATA_P_READS + POOL_TEMP_INDEX_P_READS + POOL_TEMP_XDA_P_READS + 
				POOL_COL_P_READS + POOL_TEMP_COL_P_READS) AS PHYSICAL_READS,
			SUM(DIRECT_READS) AS DIRECT_READS,
			CASE 
				WHEN SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + 
						POOL_TEMP_DATA_L_READS + POOL_TEMP_INDEX_L_READS + POOL_TEMP_XDA_L_READS + 
						POOL_COL_L_READS + POOL_TEMP_COL_L_READS + DIRECT_READS) > 0 
				THEN (SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + 
						POOL_TEMP_DATA_P_READS + POOL_TEMP_INDEX_P_READS + POOL_TEMP_XDA_P_READS + 
						POOL_COL_P_READS + POOL_TEMP_COL_P_READS + DIRECT_READS) * 1.0) / 
					SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS + 
						POOL_TEMP_DATA_L_READS + POOL_TEMP_INDEX_L_READS + POOL_TEMP_XDA_L_READS + 
						POOL_COL_L_READS + POOL_TEMP_COL_L_READS + DIRECT_READS)
				ELSE 0
			END AS HIT_RATIO
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	// Time spent queries
	querySimpleAvgPoolReadTime = `
		SELECT 
			CASE 
				WHEN SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) > 0 
				THEN SUM(POOL_READ_TIME) / SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS)
				ELSE 0
			END AS AVG_POOL_READ_TIME
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimpleAvgDirectReadTime = `
		SELECT 
			CASE 
				WHEN SUM(DIRECT_READS) > 0 
				THEN SUM(DIRECT_READ_TIME) / SUM(DIRECT_READS)
				ELSE 0
			END AS AVG_DIRECT_READ_TIME
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleAvgPoolWriteTime = `
		SELECT 
			CASE 
				WHEN SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) > 0 
				THEN SUM(POOL_WRITE_TIME) / SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES)
				ELSE 0
			END AS AVG_POOL_WRITE_TIME
		FROM TABLE(MON_GET_BUFFERPOOL('', -2))
	`
	querySimpleAvgDirectWriteTime = `
		SELECT 
			CASE 
				WHEN SUM(DIRECT_WRITES) > 0 
				THEN SUM(DIRECT_WRITE_TIME) / SUM(DIRECT_WRITES)
				ELSE 0
			END AS AVG_DIRECT_WRITE_TIME
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleSorts = `
		SELECT SUM(TOTAL_SORTS) AS SORTS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleSortOverflows = `
		SELECT SUM(SORT_OVERFLOWS) AS SORT_OVERFLOWS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleHashJoins = `
		SELECT SUM(TOTAL_HASH_JOINS) AS HASH_JOINS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleHashJoinOverflows = `
		SELECT SUM(HASH_JOIN_OVERFLOWS) AS HASH_JOIN_OVERFLOWS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleHashGrpbys = `
		SELECT SUM(TOTAL_HASH_GRPBYS) AS HASH_GRPBYS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimpleHashGrpbyOverflows = `
		SELECT SUM(HASH_GRPBY_OVERFLOWS) AS HASH_GRPBY_OVERFLOWS
		FROM TABLE(MON_GET_DATABASE(-2))
	`
	querySimplePctTimeQueued = `
		SELECT 
			CASE 
				WHEN SUM(TOTAL_RQST_TIME) > 0 
				THEN (SUM(WLM_QUEUE_TIME_TOTAL) * 100.0) / SUM(TOTAL_RQST_TIME)
				ELSE 0
			END AS PCT_TIME_QUEUED
		FROM TABLE(MON_GET_SERVICE_SUBCLASS(NULL, NULL, -2))
	`

	// Screen 15: Prefetchers queries
	// Uses simple individual queries from dcmtop
	queryPrefetcherMetrics = `
		SELECT 
			BP_NAME AS BUFFERPOOL_NAME,
			CASE 
				WHEN (POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + 
					  POOL_COL_P_READS + POOL_TEMP_DATA_P_READS + POOL_TEMP_INDEX_P_READS + 
					  POOL_TEMP_XDA_P_READS + POOL_TEMP_COL_P_READS) > 0 
				THEN DECIMAL((POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS + 
							  POOL_ASYNC_XDA_READS + POOL_ASYNC_COL_READS) * 100.0 / 
							 (POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + 
							  POOL_COL_P_READS + POOL_TEMP_DATA_P_READS + POOL_TEMP_INDEX_P_READS + 
							  POOL_TEMP_XDA_P_READS + POOL_TEMP_COL_P_READS), 5, 2) 
				ELSE 0.00 
			END AS PREFETCH_RATIO_PCT,
			CASE 
				WHEN (POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES + POOL_COL_WRITES) > 0 
				THEN DECIMAL((POOL_ASYNC_DATA_WRITES + POOL_ASYNC_INDEX_WRITES + 
							  POOL_ASYNC_XDA_WRITES + POOL_ASYNC_COL_WRITES) * 100.0 / 
							 (POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES + POOL_COL_WRITES), 5, 2) 
				ELSE 0.00 
			END AS CLEANER_RATIO_PCT,
			(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS + POOL_COL_P_READS + 
			 POOL_TEMP_DATA_P_READS + POOL_TEMP_INDEX_P_READS + POOL_TEMP_XDA_P_READS + 
			 POOL_TEMP_COL_P_READS) AS PHYSICAL_READS,
			(POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS + POOL_ASYNC_XDA_READS + 
			 POOL_ASYNC_COL_READS) AS ASYNCHRONOUS_READS,
			CASE 
				WHEN PREFETCH_WAITS > 0 
				THEN DECIMAL(PREFETCH_WAIT_TIME / PREFETCH_WAITS, 10, 2) 
				ELSE 0.00 
			END AS PREFETCH_WAITS_TIME_MS,
			UNREAD_PREFETCH_PAGES
		FROM TABLE(MON_GET_BUFFERPOOL('', -2)) 
		ORDER BY BP_NAME`

	// SNAP-based simple queries for older DB2 versions
	// These are fallback queries when MON_GET functions are not available

	// Database status from SNAP
	querySnapDatabaseStatus = `SELECT DB_STATUS AS DATABASE_STATUS FROM SYSIBMADM.SNAPDB FETCH FIRST 1 ROW ONLY`

	// Connection metrics from SNAP
	querySnapConnectionsActive = `SELECT COUNT(*) AS ACTIVE_CONNECTIONS FROM SYSIBMADM.APPLICATIONS WHERE APPL_STATUS = 'CONNECTED'`
	querySnapConnectionsTotal  = `SELECT COUNT(*) AS TOTAL_CONNECTIONS FROM SYSIBMADM.APPLICATIONS`

	// Throughput from SNAP
	querySnapTransactions  = `SELECT SUM(TOTAL_APP_COMMITS + INT_COMMITS + TOTAL_APP_ROLLBACKS + INT_ROLLBACKS) AS TRANSACTIONS FROM SYSIBMADM.SNAPDB`
	querySnapSelectStmts   = `SELECT SUM(SELECT_SQL_STMTS) AS SELECT_STMTS FROM SYSIBMADM.SNAPDB`
	querySnapUIDStmts      = `SELECT SUM(UID_SQL_STMTS) AS UID_STMTS FROM SYSIBMADM.SNAPDB`
	querySnapLogReads      = `SELECT SUM(LOG_READS) AS LOG_READS FROM SYSIBMADM.SNAPDB`
	querySnapLogWrites     = `SELECT SUM(LOG_WRITES) AS LOG_WRITES FROM SYSIBMADM.SNAPDB`
	querySnapLogicalReads  = `SELECT SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS + POOL_XDA_L_READS) AS LOGICAL_READS FROM SYSIBMADM.SNAPBP`
	querySnapPhysicalReads = `SELECT SUM(POOL_DATA_P_READS + POOL_INDEX_P_READS + POOL_XDA_P_READS) AS PHYSICAL_READS FROM SYSIBMADM.SNAPBP`
	querySnapAsyncReads    = `SELECT SUM(POOL_ASYNC_DATA_READS + POOL_ASYNC_INDEX_READS + POOL_ASYNC_XDA_READS) AS ASYNC_READS FROM SYSIBMADM.SNAPBP`
	querySnapWrites        = `SELECT SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES + POOL_XDA_WRITES) AS WRITES FROM SYSIBMADM.SNAPBP`
	querySnapAsyncWrites   = `SELECT SUM(POOL_ASYNC_DATA_WRITES + POOL_ASYNC_INDEX_WRITES + POOL_ASYNC_XDA_WRITES) AS ASYNC_WRITES FROM SYSIBMADM.SNAPBP`

	// Lock metrics from SNAP
	querySnapLockHeld        = `SELECT SUM(LOCKS_HELD) AS LOCKS_HELD FROM SYSIBMADM.SNAPDB`
	querySnapLockWaits       = `SELECT SUM(LOCK_WAITS) AS LOCK_WAITS FROM SYSIBMADM.SNAPDB`
	querySnapLockTimeouts    = `SELECT SUM(LOCK_TIMEOUTS) AS LOCK_TIMEOUTS FROM SYSIBMADM.SNAPDB`
	querySnapLockEscalations = `SELECT SUM(LOCK_ESCALS) AS LOCK_ESCALATIONS FROM SYSIBMADM.SNAPDB`
	querySnapDeadlocks       = `SELECT SUM(DEADLOCKS) AS DEADLOCKS FROM SYSIBMADM.SNAPDB`

	// Time spent from SNAP
	querySnapAvgPoolReadTime = `
		SELECT 
			CASE 
				WHEN SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS) > 0 
				THEN SUM(POOL_READ_TIME) / SUM(POOL_DATA_L_READS + POOL_INDEX_L_READS)
				ELSE 0
			END AS AVG_POOL_READ_TIME
		FROM SYSIBMADM.SNAPBP
	`
	querySnapAvgDirectReadTime = `
		SELECT 
			CASE 
				WHEN SUM(DIRECT_READS) > 0 
				THEN SUM(DIRECT_READ_TIME) / SUM(DIRECT_READS)
				ELSE 0
			END AS AVG_DIRECT_READ_TIME
		FROM SYSIBMADM.SNAPDB
	`
	querySnapAvgPoolWriteTime = `
		SELECT 
			CASE 
				WHEN SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES) > 0 
				THEN SUM(POOL_WRITE_TIME) / SUM(POOL_DATA_WRITES + POOL_INDEX_WRITES)
				ELSE 0
			END AS AVG_POOL_WRITE_TIME
		FROM SYSIBMADM.SNAPBP
	`
	querySnapAvgDirectWriteTime = `
		SELECT 
			CASE 
				WHEN SUM(DIRECT_WRITES) > 0 
				THEN SUM(DIRECT_WRITE_TIME) / SUM(DIRECT_WRITES)
				ELSE 0
			END AS AVG_DIRECT_WRITE_TIME
		FROM SYSIBMADM.SNAPDB
	`
	querySnapSorts         = `SELECT SUM(TOTAL_SORTS) AS SORTS FROM SYSIBMADM.SNAPDB`
	querySnapSortOverflows = `SELECT SUM(SORT_OVERFLOWS) AS SORT_OVERFLOWS FROM SYSIBMADM.SNAPDB`
)
