import { Cell } from '@jupyterlab/cells';
import { CodeEditor, JSONEditor } from '@jupyterlab/codeeditor';
import { ObservableJSON } from '@jupyterlab/observables';
import { ITranslator } from '@jupyterlab/translation';
import { ReadonlyPartialJSONValue } from '@lumino/coreutils';
import { ConflatableMessage, Message } from '@lumino/messaging';
import { Widget } from '@lumino/widgets';
import { NotebookPanel } from './panel';
import { INotebookTools, INotebookTracker } from './tokens';
declare class RankedPanel<T extends Widget = Widget> extends Widget {
    constructor();
    addWidget(widget: Widget, rank: number): void;
    /**
     * Handle the removal of a child
     *
     */
    protected onChildRemoved(msg: Widget.ChildMessage): void;
    private _items;
}
/**
 * A widget that provides metadata tools.
 */
export declare class NotebookTools extends Widget implements INotebookTools {
    /**
     * Construct a new NotebookTools object.
     */
    constructor(options: NotebookTools.IOptions);
    /**
     * The active cell widget.
     */
    get activeCell(): Cell | null;
    /**
     * The currently selected cells.
     */
    get selectedCells(): Cell[];
    /**
     * The current notebook.
     */
    get activeNotebookPanel(): NotebookPanel | null;
    /**
     * Add a cell tool item.
     */
    addItem(options: NotebookTools.IAddOptions): void;
    addSection(options: NotebookTools.IAddSectionOptions): void;
    /**
     * Handle a change to the notebook panel.
     */
    private _onActiveNotebookPanelChanged;
    /**
     * Handle a change to the active cell.
     */
    private _onActiveCellChanged;
    /**
     * Handle a change in the selection.
     */
    private _onSelectionChanged;
    /**
     * Handle a change in the active cell metadata.
     */
    private _onActiveNotebookPanelMetadataChanged;
    /**
     * Handle a change in the notebook model metadata.
     */
    private _onActiveCellMetadataChanged;
    private _toolChildren;
    translator: ITranslator;
    private _tools;
    private _tracker;
    private _prevActiveCell;
    private _prevActiveNotebookModel;
}
/**
 * The namespace for NotebookTools class statics.
 */
export declare namespace NotebookTools {
    /**
     * A type alias for a readonly partial JSON tuples `[option, value]`.
     * `option` should be localized.
     *
     * Note: Partial here means that JSON object attributes can be `undefined`.
     */
    type ReadonlyPartialJSONOptionValueArray = [
        ReadonlyPartialJSONValue | undefined,
        ReadonlyPartialJSONValue
    ][];
    /**
     * Interface for an extended panel section.
     */
    interface IToolPanel {
        /**
         * The name of the section.
         */
        section: string;
        /**
         * The associated panel, only one for a section.
         */
        panel: RankedPanel<NotebookTools.Tool>;
        /**
         * The rank of the section on the notebooktools panel.
         */
        rank?: number | null;
    }
    /**
     * The options used to create a NotebookTools object.
     */
    interface IOptions {
        /**
         * The notebook tracker used by the notebook tools.
         */
        tracker: INotebookTracker;
        /**
         * Language translator.
         */
        translator?: ITranslator;
    }
    /**
     * The options used to add an item to the notebook tools.
     */
    interface IAddOptions {
        /**
         * The tool to add to the notebook tools area.
         */
        tool: INotebookTools.ITool;
        /**
         * The section to which the tool should be added.
         */
        section: string;
        /**
         * The rank order of the widget among its siblings.
         */
        rank?: number;
    }
    /**
     * The options used to add a section to the notebook tools.
     */
    interface IAddSectionOptions {
        /**
         * The name of the new section.
         */
        sectionName: string;
        /**
         * The tool to add to the notebook tools area.
         */
        tool?: INotebookTools.ITool;
        /**
         * The label of the new section.
         */
        label?: string;
        /**
         * The rank order of the section among its siblings.
         */
        rank?: number;
    }
    /**
     * A singleton conflatable `'activenotebookpanel-changed'` message.
     */
    const ActiveNotebookPanelMessage: ConflatableMessage;
    /**
     * A singleton conflatable `'activecell-changed'` message.
     */
    const ActiveCellMessage: ConflatableMessage;
    /**
     * A singleton conflatable `'selection-changed'` message.
     */
    const SelectionMessage: ConflatableMessage;
    /**
     * The base notebook tool, meant to be subclassed.
     */
    class Tool extends Widget implements INotebookTools.ITool {
        /**
         * The notebook tools object.
         */
        notebookTools: INotebookTools;
        dispose(): void;
        /**
         * Process a message sent to the widget.
         *
         * @param msg - The message sent to the widget.
         */
        processMessage(msg: Message): void;
        /**
         * Handle a change to the notebook panel.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        protected onActiveNotebookPanelChanged(msg: Message): void;
        /**
         * Handle a change to the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        protected onActiveCellChanged(msg: Message): void;
        /**
         * Handle a change to the selection.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        protected onSelectionChanged(msg: Message): void;
        /**
         * Handle a change to the metadata of the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        protected onActiveCellMetadataChanged(msg: ObservableJSON.ChangeMessage): void;
        /**
         * Handle a change to the metadata of the active cell.
         *
         * #### Notes
         * The default implementation is a no-op.
         */
        protected onActiveNotebookPanelMetadataChanged(msg: ObservableJSON.ChangeMessage): void;
    }
    /**
     * A raw metadata editor.
     */
    class MetadataEditorTool extends Tool {
        /**
         * Construct a new raw metadata tool.
         */
        constructor(options: MetadataEditorTool.IOptions);
        /**
         * The editor used by the tool.
         */
        get editor(): JSONEditor;
        /**
         * Handle a change to the notebook.
         */
        protected onActiveNotebookPanelChanged(msg: Message): void;
        protected createEditor(): void;
        private _editor;
        private _editorLabel;
        private _editorFactory;
    }
    /**
     * The namespace for `MetadataEditorTool` static data.
     */
    namespace MetadataEditorTool {
        /**
         * The options used to initialize a metadata editor tool.
         */
        interface IOptions {
            /**
             * The editor factory used by the tool.
             */
            editorFactory: CodeEditor.Factory;
            /**
             * The label for the JSON editor
             */
            label?: string;
            /**
             * Initial collapse state, defaults to true.
             */
            collapsed?: boolean;
            /**
             * Language translator.
             */
            translator?: ITranslator;
        }
    }
}
export {};
