// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { nullTranslator } from '@jupyterlab/translation';
import { caretDownEmptyThinIcon, caretDownIcon, caretRightIcon, caretUpEmptyThinIcon, caseSensitiveIcon, classes, closeIcon, filterDotIcon, filterIcon, regexIcon, VDomRenderer, wordIcon } from '@jupyterlab/ui-components';
import { Signal } from '@lumino/signaling';
import { UseSignal } from '@jupyterlab/apputils';
import * as React from 'react';
import { useCallback, useEffect, useState } from 'react';
const OVERLAY_CLASS = 'jp-DocumentSearch-overlay';
const OVERLAY_ROW_CLASS = 'jp-DocumentSearch-overlay-row';
const INPUT_CLASS = 'jp-DocumentSearch-input';
const INPUT_LABEL_CLASS = 'jp-DocumentSearch-input-label';
const INPUT_WRAPPER_CLASS = 'jp-DocumentSearch-input-wrapper';
const INPUT_BUTTON_CLASS_OFF = 'jp-DocumentSearch-input-button-off';
const INPUT_BUTTON_CLASS_ON = 'jp-DocumentSearch-input-button-on';
const INDEX_COUNTER_CLASS = 'jp-DocumentSearch-index-counter';
const UP_DOWN_BUTTON_WRAPPER_CLASS = 'jp-DocumentSearch-up-down-wrapper';
const UP_DOWN_BUTTON_CLASS = 'jp-DocumentSearch-up-down-button';
const FILTER_BUTTON_CLASS = 'jp-DocumentSearch-filter-button';
const FILTER_BUTTON_ENABLED_CLASS = 'jp-DocumentSearch-filter-button-enabled';
const REGEX_ERROR_CLASS = 'jp-DocumentSearch-regex-error';
const SEARCH_OPTIONS_CLASS = 'jp-DocumentSearch-search-options';
const SEARCH_FILTER_DISABLED_CLASS = 'jp-DocumentSearch-search-filter-disabled';
const SEARCH_FILTER_CLASS = 'jp-DocumentSearch-search-filter';
const REPLACE_BUTTON_CLASS = 'jp-DocumentSearch-replace-button';
const REPLACE_BUTTON_WRAPPER_CLASS = 'jp-DocumentSearch-replace-button-wrapper';
const REPLACE_WRAPPER_CLASS = 'jp-DocumentSearch-replace-wrapper-class';
const REPLACE_TOGGLE_CLASS = 'jp-DocumentSearch-replace-toggle';
const TOGGLE_WRAPPER = 'jp-DocumentSearch-toggle-wrapper';
const TOGGLE_PLACEHOLDER = 'jp-DocumentSearch-toggle-placeholder';
const BUTTON_CONTENT_CLASS = 'jp-DocumentSearch-button-content';
const BUTTON_WRAPPER_CLASS = 'jp-DocumentSearch-button-wrapper';
const SPACER_CLASS = 'jp-DocumentSearch-spacer';
function SearchInput(props) {
    const [rows, setRows] = useState(1);
    const updateDimensions = useCallback((event) => {
        var _a;
        const element = event
            ? event.target
            : (_a = props.inputRef) === null || _a === void 0 ? void 0 : _a.current;
        if (element) {
            const split = element.value.split(/\n/);
            // use the longest string out of all lines to compute the width.
            let longest = split.reduce((a, b) => (a.length > b.length ? a : b), '');
            if (element.parentNode && element.parentNode instanceof HTMLElement) {
                element.parentNode.dataset.value = longest;
            }
            setRows(split.length);
        }
    }, []);
    useEffect(() => {
        var _a, _b;
        // For large part, `focusSearchInput()` is responsible for focusing and
        // selecting the search input, however when `initialValue` changes, this
        // triggers React re-render to update `defaultValue` (implemented via `key`)
        // which means that `focusSearchInput` is no longer effective as it has
        // already fired before the re-render, hence we use this conditional effect.
        (_b = (_a = props.inputRef) === null || _a === void 0 ? void 0 : _a.current) === null || _b === void 0 ? void 0 : _b.select();
        // After any change to initial value we also want to update rows in case if
        // multi-line text was selected.
        updateDimensions();
    }, [props.initialValue]);
    return (React.createElement("label", { className: INPUT_LABEL_CLASS },
        React.createElement("textarea", { onChange: e => {
                props.onChange(e);
                updateDimensions(e);
            }, onKeyDown: e => {
                props.onKeyDown(e);
                updateDimensions(e);
            }, rows: rows, placeholder: props.placeholder, className: INPUT_CLASS, 
            // Setting a key ensures that `defaultValue` will become updated
            // when the initial value changes.
            key: props.autoUpdate ? props.initialValue : null, tabIndex: 0, ref: props.inputRef, title: props.title, defaultValue: props.initialValue, autoFocus: props.autoFocus })));
}
function SearchEntry(props) {
    var _a;
    const trans = ((_a = props.translator) !== null && _a !== void 0 ? _a : nullTranslator).load('jupyterlab');
    const caseButtonToggleClass = classes(props.caseSensitive ? INPUT_BUTTON_CLASS_ON : INPUT_BUTTON_CLASS_OFF, BUTTON_CONTENT_CLASS);
    const regexButtonToggleClass = classes(props.useRegex ? INPUT_BUTTON_CLASS_ON : INPUT_BUTTON_CLASS_OFF, BUTTON_CONTENT_CLASS);
    const wordButtonToggleClass = classes(props.wholeWords ? INPUT_BUTTON_CLASS_ON : INPUT_BUTTON_CLASS_OFF, BUTTON_CONTENT_CLASS);
    const wrapperClass = INPUT_WRAPPER_CLASS;
    return (React.createElement("div", { className: wrapperClass },
        React.createElement(SearchInput, { placeholder: trans.__('Find'), onChange: e => props.onChange(e), onKeyDown: e => props.onKeydown(e), inputRef: props.inputRef, initialValue: props.initialSearchText, title: trans.__('Find'), autoFocus: true, autoUpdate: true }),
        React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => {
                props.onCaseSensitiveToggled();
            }, tabIndex: 0, title: trans.__('Match Case') },
            React.createElement(caseSensitiveIcon.react, { className: caseButtonToggleClass, tag: "span" })),
        React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.onWordToggled(), tabIndex: 0, title: trans.__('Match Whole Word') },
            React.createElement(wordIcon.react, { className: wordButtonToggleClass, tag: "span" })),
        React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.onRegexToggled(), tabIndex: 0, title: trans.__('Use Regular Expression') },
            React.createElement(regexIcon.react, { className: regexButtonToggleClass, tag: "span" }))));
}
function ReplaceEntry(props) {
    var _a, _b, _c;
    const trans = ((_a = props.translator) !== null && _a !== void 0 ? _a : nullTranslator).load('jupyterlab');
    const preserveCaseButtonToggleClass = classes(props.preserveCase ? INPUT_BUTTON_CLASS_ON : INPUT_BUTTON_CLASS_OFF, BUTTON_CONTENT_CLASS);
    return (React.createElement("div", { className: REPLACE_WRAPPER_CLASS },
        React.createElement("div", { className: INPUT_WRAPPER_CLASS },
            React.createElement(SearchInput, { placeholder: trans.__('Replace'), initialValue: (_b = props.replaceText) !== null && _b !== void 0 ? _b : '', onKeyDown: e => props.onReplaceKeydown(e), onChange: e => props.onChange(e), title: trans.__('Replace'), autoFocus: false, autoUpdate: false }),
            ((_c = props.replaceOptionsSupport) === null || _c === void 0 ? void 0 : _c.preserveCase) ? (React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.onPreserveCaseToggled(), tabIndex: 0, title: trans.__('Preserve Case') },
                React.createElement(caseSensitiveIcon.react, { className: preserveCaseButtonToggleClass, tag: "span" }))) : null),
        React.createElement("button", { className: REPLACE_BUTTON_WRAPPER_CLASS, onClick: () => props.onReplaceCurrent(), tabIndex: 0 },
            React.createElement("span", { className: `${REPLACE_BUTTON_CLASS} ${BUTTON_CONTENT_CLASS}`, tabIndex: 0 }, trans.__('Replace'))),
        React.createElement("button", { className: REPLACE_BUTTON_WRAPPER_CLASS, tabIndex: 0, onClick: () => props.onReplaceAll() },
            React.createElement("span", { className: `${REPLACE_BUTTON_CLASS} ${BUTTON_CONTENT_CLASS}`, tabIndex: -1 }, trans.__('Replace All')))));
}
function UpDownButtons(props) {
    return (React.createElement("div", { className: UP_DOWN_BUTTON_WRAPPER_CLASS },
        React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.onHighlightPrevious(), tabIndex: 0, title: props.trans.__('Previous Match') },
            React.createElement(caretUpEmptyThinIcon.react, { className: classes(UP_DOWN_BUTTON_CLASS, BUTTON_CONTENT_CLASS), tag: "span" })),
        React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.onHighlightNext(), tabIndex: 0, title: props.trans.__('Next Match') },
            React.createElement(caretDownEmptyThinIcon.react, { className: classes(UP_DOWN_BUTTON_CLASS, BUTTON_CONTENT_CLASS), tag: "span" }))));
}
function SearchIndices(props) {
    return (React.createElement("div", { className: INDEX_COUNTER_CLASS }, props.totalMatches === 0
        ? '-/-'
        : `${props.currentIndex === null ? '-' : props.currentIndex + 1}/${props.totalMatches}`));
}
function FilterToggle(props) {
    let className = `${FILTER_BUTTON_CLASS} ${BUTTON_CONTENT_CLASS}`;
    if (props.visible) {
        className = `${className} ${FILTER_BUTTON_ENABLED_CLASS}`;
    }
    const icon = props.anyEnabled ? filterDotIcon : filterIcon;
    return (React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => props.toggleVisible(), tabIndex: 0, title: props.visible
            ? props.trans.__('Hide Search Filters')
            : props.trans.__('Show Search Filters') },
        React.createElement(icon.react, { className: className, tag: "span", height: "20px", width: "20px" })));
}
function FilterSelection(props) {
    return (React.createElement("label", { className: props.isEnabled
            ? SEARCH_FILTER_CLASS
            : `${SEARCH_FILTER_CLASS} ${SEARCH_FILTER_DISABLED_CLASS}`, title: props.description },
        React.createElement("input", { type: "checkbox", className: "jp-mod-styled", disabled: !props.isEnabled, checked: props.value, onChange: props.onToggle }),
        props.title));
}
class SearchOverlay extends React.Component {
    constructor(props) {
        super(props);
        this.translator = props.translator || nullTranslator;
    }
    _onSearchChange(event) {
        const searchText = event.target.value;
        this.props.onSearchChanged(searchText);
    }
    _onSearchKeydown(event) {
        if (event.keyCode === 13) {
            // Enter pressed
            event.stopPropagation();
            event.preventDefault();
            if (event.ctrlKey) {
                const textarea = event.target;
                this._insertNewLine(textarea);
                this.props.onSearchChanged(textarea.value);
            }
            else {
                event.shiftKey
                    ? this.props.onHighlightPrevious()
                    : this.props.onHighlightNext();
            }
        }
    }
    _onReplaceKeydown(event) {
        if (event.keyCode === 13) {
            // Enter pressed
            event.stopPropagation();
            event.preventDefault();
            if (event.ctrlKey) {
                this._insertNewLine(event.target);
            }
            else {
                this.props.onReplaceCurrent();
            }
        }
    }
    _insertNewLine(textarea) {
        const [start, end] = [textarea.selectionStart, textarea.selectionEnd];
        textarea.setRangeText('\n', start, end, 'end');
    }
    _onClose() {
        // Clean up and close widget.
        this.props.onClose();
    }
    _onReplaceToggled() {
        // Deactivate invalid replace filters
        if (!this.props.replaceEntryVisible) {
            for (const key in this.props.filtersDefinition) {
                const filter = this.props.filtersDefinition[key];
                if (!filter.supportReplace) {
                    this.props.onFilterChanged(key, false).catch(reason => {
                        console.error(`Fail to update filter value for ${filter.title}:\n${reason}`);
                    });
                }
            }
        }
        this.props.onReplaceEntryShown(!this.props.replaceEntryVisible);
    }
    _toggleFiltersVisibility() {
        this.props.onFiltersVisibilityChanged(!this.props.filtersVisible);
    }
    render() {
        var _a;
        const trans = this.translator.load('jupyterlab');
        const showReplace = !this.props.isReadOnly && this.props.replaceEntryVisible;
        const filters = this.props.filtersDefinition;
        const hasFilters = Object.keys(filters).length > 0;
        const filterToggle = hasFilters ? (React.createElement(FilterToggle, { visible: this.props.filtersVisible, anyEnabled: Object.keys(filters).some(name => {
                var _a;
                const filter = filters[name];
                return (_a = this.props.filters[name]) !== null && _a !== void 0 ? _a : filter.default;
            }), toggleVisible: () => this._toggleFiltersVisibility(), trans: trans })) : null;
        const filter = hasFilters ? (React.createElement("div", { className: SEARCH_OPTIONS_CLASS }, Object.keys(filters).map(name => {
            var _a;
            const filter = filters[name];
            return (React.createElement(FilterSelection, { key: name, title: filter.title, description: filter.description, isEnabled: !showReplace || filter.supportReplace, onToggle: async () => {
                    await this.props.onFilterChanged(name, !this.props.filters[name]);
                }, value: (_a = this.props.filters[name]) !== null && _a !== void 0 ? _a : filter.default }));
        }))) : null;
        const icon = this.props.replaceEntryVisible
            ? caretDownIcon
            : caretRightIcon;
        // TODO: Error messages from regex are not currently localizable.
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { className: OVERLAY_ROW_CLASS },
                this.props.isReadOnly ? (React.createElement("div", { className: TOGGLE_PLACEHOLDER })) : (React.createElement("button", { className: TOGGLE_WRAPPER, onClick: () => this._onReplaceToggled(), tabIndex: 0, title: trans.__('Toggle Replace') },
                    React.createElement(icon.react, { className: `${REPLACE_TOGGLE_CLASS} ${BUTTON_CONTENT_CLASS}`, tag: "span", elementPosition: "center", height: "20px", width: "20px" }))),
                React.createElement(SearchEntry, { inputRef: this.props.searchInputRef, useRegex: this.props.useRegex, caseSensitive: this.props.caseSensitive, wholeWords: this.props.wholeWords, onCaseSensitiveToggled: this.props.onCaseSensitiveToggled, onRegexToggled: this.props.onRegexToggled, onWordToggled: this.props.onWordToggled, onKeydown: (e) => this._onSearchKeydown(e), onChange: (e) => this._onSearchChange(e), initialSearchText: this.props.initialSearchText, translator: this.translator }),
                filterToggle,
                React.createElement(SearchIndices, { currentIndex: this.props.currentIndex, totalMatches: (_a = this.props.totalMatches) !== null && _a !== void 0 ? _a : 0 }),
                React.createElement(UpDownButtons, { onHighlightPrevious: () => {
                        this.props.onHighlightPrevious();
                    }, onHighlightNext: () => {
                        this.props.onHighlightNext();
                    }, trans: trans }),
                React.createElement("button", { className: BUTTON_WRAPPER_CLASS, onClick: () => this._onClose(), tabIndex: 0 },
                    React.createElement(closeIcon.react, { className: "jp-icon-hover", elementPosition: "center", height: "16px", width: "16px" }))),
            React.createElement("div", { className: OVERLAY_ROW_CLASS }, showReplace ? (React.createElement(React.Fragment, null,
                React.createElement(ReplaceEntry, { onPreserveCaseToggled: this.props.onPreserveCaseToggled, onReplaceKeydown: (e) => this._onReplaceKeydown(e), onChange: (e) => this.props.onReplaceChanged(e.target.value), onReplaceCurrent: () => this.props.onReplaceCurrent(), onReplaceAll: () => this.props.onReplaceAll(), replaceOptionsSupport: this.props.replaceOptionsSupport, replaceText: this.props.replaceText, preserveCase: this.props.preserveCase, translator: this.translator }),
                React.createElement("div", { className: SPACER_CLASS }))) : null),
            this.props.filtersVisible ? filter : null,
            !!this.props.errorMessage && (React.createElement("div", { className: REGEX_ERROR_CLASS }, this.props.errorMessage))));
    }
}
/**
 * Search document widget
 */
export class SearchDocumentView extends VDomRenderer {
    /**
     * Search document widget constructor.
     *
     * @param model Search document model
     * @param translator Application translator object
     */
    constructor(model, translator) {
        super(model);
        this.translator = translator;
        this._showReplace = false;
        this._showFilters = false;
        this._closed = new Signal(this);
        this.addClass(OVERLAY_CLASS);
        this._searchInput = React.createRef();
    }
    /**
     * A signal emitted when the widget is closed.
     *
     * Closing the widget detached it from the DOM but does not dispose it.
     */
    get closed() {
        return this._closed;
    }
    /**
     * Focus search input.
     */
    focusSearchInput() {
        var _a;
        (_a = this._searchInput.current) === null || _a === void 0 ? void 0 : _a.select();
    }
    /**
     * Set the initial search text.
     */
    setSearchText(search) {
        this.model.initialQuery = search;
        // Only set the new search text to search expression if there is any
        // to avoid nullifying the one that was remembered from last time.
        if (search) {
            this.model.searchExpression = search;
        }
    }
    /**
     * Set the replace text
     *
     * It does not trigger a view update.
     */
    setReplaceText(replace) {
        this.model.replaceText = replace;
    }
    /**
     * Show the replacement input box.
     */
    showReplace() {
        this.setReplaceInputVisibility(true);
    }
    /**
     * A message handler invoked on a `'close-request'` message.
     *
     * #### Notes
     * On top of the default implementation emit closed signal and end model query.
     */
    onCloseRequest(msg) {
        super.onCloseRequest(msg);
        this._closed.emit();
        void this.model.endQuery();
    }
    setReplaceInputVisibility(v) {
        if (this._showReplace !== v) {
            this._showReplace = v;
            this.update();
        }
    }
    setFiltersVisibility(v) {
        if (this._showFilters !== v) {
            this._showFilters = v;
            this.update();
        }
    }
    render() {
        return this.model.filtersDefinitionChanged ? (React.createElement(UseSignal, { signal: this.model.filtersDefinitionChanged }, () => this._renderOverlay())) : (this._renderOverlay());
    }
    _renderOverlay() {
        return (React.createElement(SearchOverlay, { caseSensitive: this.model.caseSensitive, currentIndex: this.model.currentIndex, isReadOnly: this.model.isReadOnly, errorMessage: this.model.parsingError, filters: this.model.filters, filtersDefinition: this.model.filtersDefinition, preserveCase: this.model.preserveCase, replaceEntryVisible: this._showReplace, filtersVisible: this._showFilters, replaceOptionsSupport: this.model.replaceOptionsSupport, replaceText: this.model.replaceText, initialSearchText: this.model.initialQuery, searchInputRef: this._searchInput, totalMatches: this.model.totalMatches, translator: this.translator, useRegex: this.model.useRegex, wholeWords: this.model.wholeWords, onCaseSensitiveToggled: () => {
                this.model.caseSensitive = !this.model.caseSensitive;
            }, onRegexToggled: () => {
                this.model.useRegex = !this.model.useRegex;
            }, onWordToggled: () => {
                this.model.wholeWords = !this.model.wholeWords;
            }, onFilterChanged: async (name, value) => {
                await this.model.setFilter(name, value);
            }, onFiltersVisibilityChanged: (v) => {
                this.setFiltersVisibility(v);
            }, onHighlightNext: () => {
                void this.model.highlightNext();
            }, onHighlightPrevious: () => {
                void this.model.highlightPrevious();
            }, onPreserveCaseToggled: () => {
                this.model.preserveCase = !this.model.preserveCase;
            }, onSearchChanged: (q) => {
                this.model.searchExpression = q;
            }, onClose: () => {
                this.close();
            }, onReplaceEntryShown: (v) => {
                this.setReplaceInputVisibility(v);
            }, onReplaceChanged: (q) => {
                this.model.replaceText = q;
            }, onReplaceCurrent: () => {
                void this.model.replaceCurrentMatch();
            }, onReplaceAll: () => {
                void this.model.replaceAllMatches();
            } }));
    }
}
//# sourceMappingURL=searchview.js.map